# -*- coding: ascii -*-

###########################################################################
# clive, video extraction utility
# Copyright (C) 2007-2008 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

## Tkinter based classes for GUI history browsing.

import Tkinter
import tkMessageBox
import csv
import errno

from clive.opts import Options

__all__ = ['History']


## The main GUI class for history browsing.
class History:

	## Creates the widgets and enters the main loop.
	#
	# Displays the Tkinter based GUI.
	#
	# \return A list of selected URLs.
	def main(self):
		self._entries = []
		self._widgets = {}
		self._selected_urls = []
		self._changed = 0

		self._root = Tkinter.Tk()
		self._root.title('clive - history')
		self._root.geometry('+100+100')
		self._create_widgets()
		self._load_urls()
		self._root.mainloop()

		return self._selected_urls

	def _create_widgets(self):
		self._root.rowconfigure(0, weight=1)
		self._root.columnconfigure(0, weight=1)

		frame = Tkinter.Frame(self._root)
		frame.grid(row=0, column=0, padx=10, pady=10, sticky='nwse')
		main_frame = frame

		frame.grid_rowconfigure(0, weight=1)
		frame.grid_columnconfigure(0, weight=1)

		# Listbox + scrollbars
		sbv = Tkinter.Scrollbar(frame, orient=Tkinter.VERTICAL)
		sbh = Tkinter.Scrollbar(frame, orient=Tkinter.HORIZONTAL)

		self._lb = Tkinter.Listbox(frame, selectmode=Tkinter.EXTENDED,
			relief=Tkinter.SUNKEN, xscrollcommand=sbh.set,
			yscrollcommand=sbv.set, width=60)

		self._lb.grid(row=0, column=0, sticky='nwse')

		self._lb.bind('<<ListboxSelect>>', self._on_lb_select)

		sbv.grid(row=0, column=1, sticky='ns')
		sbh.grid(row=1, column=0, sticky='we')

		sbv.config(command=self._lb.yview)
		sbh.config(command=self._lb.xview)

		# Label + entry
		frame = Tkinter.Frame(frame)
		frame.grid(row=2, column=0, sticky='we')

		a = [
			('entry_time', 'Time:'),
			('entry_url', 'URL:'),
			('entry_file', 'Filename:'),
			('entry_length', 'Length:'),
		]

		for (i, e) in enumerate(a):
			self._label_entry(frame, e[0], e[1], i)

		frame.grid_rowconfigure(0, weight=1)
		frame.grid_columnconfigure(1, weight=1)

		# Cancel / OK
		frame = Tkinter.Frame(main_frame)
		frame.grid(row=2, column=2, sticky='se')

		b = Tkinter.Button(frame, text='Cancel', command=self._root.destroy, width=10)
		b.grid(row=0, column=0, sticky='w', padx=5)

		b = Tkinter.Button(frame, text='OK', command=self._on_ok, width=10)
		b.grid(row=0, column=1, sticky='w')

		# Download / Delete / Clear
		frame = Tkinter.Frame(main_frame)
		frame.grid(row=0, column=2, sticky='nw', padx=10)

		a = [
			('Download Selected', self._on_extract),
			('Delete Selected', self._on_delete),
			('Clear All', self._on_clear),
		]

		for (index, btn) in enumerate(a):
			(text, cmd) = btn
			b = Tkinter.Button(frame, text=text, command=cmd, width=20)
			b.grid(row=index, column=0, sticky='w', pady=[0,5][index % 2])

		self._lb.focus()

	def _label_entry(self, parent, name, text, row, column=0):
		l = Tkinter.Label(parent, text=text)
		l.grid(row=row, column=column, pady=5, sticky='w')

		self._widgets[name] = Tkinter.StringVar()
		e = Tkinter.Entry(parent, textvariable=self._widgets[name], width=25)
		e.grid(row=row, column=column+1, pady=5, sticky='we')

	def _load_urls(self):
		opts = Options()
		try:
			reader = csv.reader(open(opts._historyfile, 'r'))
		except IOError, e:
			if e.errno == errno.ENOENT: return
			else: raise

		for row in reader:
			(datetime, url, filename, length) = row
			self._entries.append(row)
			self._lb.insert(Tkinter.END, '%s (%s)' % (filename,length))

	def _rewrite_history(self):
		opts = Options()
		try:
			f = open(opts._historyfile, 'w')
		except IOError, e:
			if e.errno == errno.ENOENT: return
			else: raise

		for row in self._entries:
			l = len(row)
			for (i, r) in enumerate(row):
				f.write('"%s"' % r)
				if i < l-1: f.write(',')
			f.write('\n')

		f.close()

	def _no_selection(self):
		tkMessageBox.showerror('No selection',
			'Please select an item from the list')
		return None

	def _on_lb_select(self, event):
		sel = map(int, self._lb.curselection())

		if len(sel) == 0:
			return

		self._set_entries(sel[0]) # Use first selected item

	def _set_entries(self, index=-1):
		a = ['entry_time', 'entry_url', 'entry_file', 'entry_length']

		for (i, e) in enumerate(a):
			try:
				text = ['',self._entries[index][i]][index != -1]
				self._widgets[e].set(text)
				#self._widgets[e].set(self._entries[index][i])
			except IndexError:
				pass

	def _on_extract(self):
		sel = self._lb.curselection()

		if len(sel) == 0:
			return self._no_selection()

		if self._changed:
			a = tkMessageBox.askquestion('Confirm', 'Save changes?',
				type=tkMessageBox.YESNOCANCEL)

			if a == tkMessageBox.YES:
				self._rewrite_history()
			elif a == tkMessageBox.NO:
				pass
			else: # Cancel
				return # NOTE: askquestion return value (cancel) seems incoherent

		for i in map(int, sel):
			self._selected_urls.append(self._entries[i][1]) # 1=url

		self._root.destroy()

	def _on_delete(self):
		sel = self._lb.curselection()

		if len(sel) == 0:
			return self._no_selection()
		
		if tkMessageBox.askquestion('Delete', 'Are you sure?',
			type=tkMessageBox.YESNO) == tkMessageBox.NO:
				return

		self._changed = 1 # !

		m = map(int, sel)
		m.reverse() # !

		for i in m:
			self._entries.pop(i)
			self._lb.delete(i)

		self._set_entries() # Reset

	def _on_clear(self):
		if self._lb.size() == 0: return

		if tkMessageBox.askquestion('Clear', 'Are you sure?',
			type=tkMessageBox.YESNO) == tkMessageBox.NO:
				return

		self._changed = 1 # !
		self._set_entries() # Reset
		self._entries = []
		self._lb.delete(0, Tkinter.END)

	def _on_ok(self):
		if self._changed:
			self._rewrite_history()
		self._root.destroy()


if __name__ == '__main__':
	History().main()


