/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * task.c --
 *
 *      Task initialization and switching routines between the host
 *      and the monitor. 
 *     
 *      A task switch: 
 *          -saves the EFLAGS,CR0,CR2,CR4, and IDT
 *          -jumps to code on the shared page
 *              which saves the registers, GDT and CR3
 *              which then restores the registers, GDT and CR3
 *          -restores the IDT,CR0,CR2,CR4 and EFLAGS
 *                
 *      This file is pretty much independent of the host OS.
 *      
 */

#ifdef linux
/* Must come before any kernel header file --hpreg */
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */

#   ifdef USE_PERFCTRS_HOSTED
#      include "perfctr.h"
#   endif

#   define EXPORT_SYMTAB
#elif defined(WINNT_DDK)
#   include <string.h>
#else
#   error "Unknown platform"
#endif

#include "vmware.h"
#include "modulecall.h"
#include "vmx86.h"
#include "task.h"
#include "vm_asm.h"
#include "cpuid.h"
#include "hostif.h"
/* On Linux, must come before any inclusion of asm/page.h --hpreg */
#include "hostKernel.h"

#if defined(_WIN64)
#include "vmmon-asm-x86-64.h"
#define USE_TEMPORARY_GDT 1
#else
#define USE_TEMPORARY_GDT 0
#endif

static uint32 dummyLVT;
static Descriptor **tempGDT;

#if defined(VM_X86_64)
static INLINE ContextInfo64 *
GetHostContext(VMCrossPage * const crosspage)
{
   return &crosspage->hostContext64;
}
#else
static INLINE ContextInfo *
GetHostContext(VMCrossPage * const crosspage)
{
   return &crosspage->hostContext;
}
#endif


static INLINE DTR *
GetHostContextGDT(VMCrossPage * const crosspage)
{
   return &GetHostContext(crosspage)->gdtr.dtr;
}


void Task_TemporaryGDTFinalize(void)
{
   if (USE_TEMPORARY_GDT) {
      const unsigned cpus = HostIF_NumOnlineLogicalCPUs();
      unsigned       i;
      
      for (i = 0; i < cpus; ++i) {
         HostIF_FreeKernelMem(tempGDT[i]);
      }
      HostIF_FreeKernelMem(tempGDT);
   }
}


Bool
Task_TemporaryGDTInitialize(void)
{
   if (USE_TEMPORARY_GDT) {
      const unsigned cpus        = HostIF_NumOnlineLogicalCPUs();
      const unsigned numPtrBytes = cpus * sizeof(Descriptor *);
      unsigned       i;
      
      /* Some linux kernels panic when allocating > 128Kb */
      ASSERT(numPtrBytes <= 131072);
      tempGDT = HostIF_AllocKernelMem(numPtrBytes, TRUE);
      if (tempGDT != NULL) {
         for (i = 0; i < cpus; ++i) {
            const unsigned bytes = 0x10000; /* maximal GDT size */
            tempGDT[i] = HostIF_AllocKernelMem(bytes, TRUE);
            if (tempGDT[i] == NULL) {
               int j;
               Warning("Unable to allocate space for temporary GDT[%d]", i);
               for (j = 0; j < i; ++j) {
                  HostIF_FreeKernelMem(tempGDT[j]);
               }
               HostIF_FreeKernelMem(tempGDT);
               return FALSE;
            }
         }
         return TRUE;
      } else {
         Warning("Unable to allocate space for temporary GDT pointers");
         return FALSE;
      }
   } else {
      return TRUE;
   }
}


/*
 *-----------------------------------------------------------------------------
 *
 * SetupTemporaryGDT --
 *
 *      Set up a temporary GDT so that the TSS 'busy bit' can be
 *      changed without affecting the host's data structures.
 *
 * Results:
 *      
 *      The host's GDT is copied (or partially copied) to the
 *      dynamically allocated temporary GDT.
 *
 *      The hostContext in the crosspage is set up to reference the
 *      host GDT or the temporary GDT.
 *
 * Side effects:
 *
 *      Crosspage modified.
 *
 * Notes:
 *
 *      An OS which checks critical data structures, such as the GDT,
 *      can fail when this module changes the TSS busy bit in the host
 *      GDT.  To avoid this problem, we use a sparse copy of the host
 *      GDT to perform the manipulation of the TSS busy bit.
 *
 *      See PR8144.
 *
 *-----------------------------------------------------------------------------
 */
static INLINE_SINGLE_CALLER void
SetupTemporaryGDT(uint32 pcpuid, VMCrossPage *crosspage,
                  DTR hostGDT, Selector cs, Selector tr)
{
   DTR *hostContextGDT = GetHostContextGDT(crosspage);

   if (USE_TEMPORARY_GDT) {
      const Bool COPY_WHOLE_GDT = TRUE;
      const VA   hostGDTVA      = HOST_KERNEL_LA_2_VA(hostGDT.offset);
      
      ASSERT((tr & SELECTOR_RPL_MASK) == 0);

      if (COPY_WHOLE_GDT) {
         memcpy(tempGDT[pcpuid], (void *)hostGDTVA, hostGDT.limit + 1); // remove string.h
      } else {
         const unsigned size = sizeof(Descriptor);
         const Selector ds   = SELECTOR_CLEAR_RPL(GET_DS()); 
         const Selector es   = SELECTOR_CLEAR_RPL(GET_ES());
         const Selector ss   = SELECTOR_CLEAR_RPL(GET_SS());

         /* %tr descriptor is two entries in 64-bit. */
         tempGDT[pcpuid][cs  / size]     = *(Descriptor *)(hostGDTVA + cs);
         tempGDT[pcpuid][ds  / size]     = *(Descriptor *)(hostGDTVA + ds);
         tempGDT[pcpuid][es  / size]     = *(Descriptor *)(hostGDTVA + es);
         tempGDT[pcpuid][ss  / size]     = *(Descriptor *)(hostGDTVA + ss);
         tempGDT[pcpuid][tr  / size]     = *(Descriptor *)(hostGDTVA + tr);
         tempGDT[pcpuid][tr  / size + 1] = *(Descriptor *)(hostGDTVA + tr + 1);
      }
      /* Set up cross page so temporary GDT will be used when
       * returning from the monitor. */
      hostContextGDT->limit  = hostGDT.limit;
      hostContextGDT->offset = (LA)HOST_KERNEL_VA_2_LA(tempGDT[pcpuid]);
   } else {
      hostContextGDT->limit  = hostGDT.limit;
      hostContextGDT->offset = hostGDT.offset;
   }
}


/*
 *------------------------------------------------------------------------------
 *
 * Task_InitCrosspage  --
 *
 *    Initialize the crosspage used to switch to the monitor task. 
 *
 * Results:
 *    0 on success
 *    != 0 on failure
 *
 * Side effects:
 *    None
 *  
 *------------------------------------------------------------------------------
 */

int 
Task_InitCrosspage(VMDriver *vm,          // IN
                   InitBlock *initParams) // IN: Initial params from the VM 
{
   Vcpuid vcpuid;

   ASSERT(sizeof(VMCrossPage) < PAGE_SIZE);
   ASSERT(MODULECALL_CROSS_PAGE_LEN == 1);
   for (vcpuid = 0; vcpuid < initParams->numVCPUs;  vcpuid++) {
      LA           linearAddr;
      void        *iparm = (void *)(uintptr_t)initParams->crosspage[vcpuid];
      VMCrossPage *p     = HostIF_MapCrossPage(vm, iparm, &linearAddr);
      MPN          crossPageMPN;

      if (p == NULL) {
         return 1;
      }

      vm->crosspage[vcpuid] = p;
      
      crossPageMPN = HostIF_LookupUserMPN(vm, iparm);

      if ((int64)crossPageMPN <= 0) {
         return 1;
      }
      
      p->crosspageMA = MPN_2_MA(crossPageMPN);
      p->hostCrossPageLA = (LA64)(uintptr_t)p;

      p->irqRelocateOffset[0]  = IRQ_HOST_INTR1_BASE; 
      p->irqRelocateOffset[1]  = IRQ_HOST_INTR2_BASE;
      p->userCallRequest       = MODULECALL_USERCALL_NONE;
      p->moduleCallInterrupted = FALSE;
      p->tscAdjustment         = 0;
   }
   return 0;
}


static Bool
DisableNMIDelivery(volatile uint32 *regPtr) // IN/OUT
{
   uint32 reg;
   
   reg = *regPtr;
   if ((APIC_LVT_DELVMODE(reg) == APIC_LVT_DELVMODE_NMI) &&
       (! APIC_LVT_ISMASKED(reg))) {
      *regPtr = reg | APIC_LVT_MASK;
      dummyLVT = *regPtr; // Force completion of masking, Bug 78470.
      return TRUE;
   }
   return FALSE;
}


static void
DisableNMI(VMDriver *vm,     // IN
           Bool *lint0NMI,   // OUT
           Bool *lint1NMI,   // OUT
           Bool *pcNMI,      // OUT
           Bool *thermalNMI) // OUT 
{
   if (vm->hostAPIC) {
      *lint0NMI = DisableNMIDelivery(&APIC_LINT0_REG(vm->hostAPIC));
      *lint1NMI = DisableNMIDelivery(&APIC_LINT1_REG(vm->hostAPIC));
      *pcNMI = DisableNMIDelivery(&APIC_PC_REG(vm->hostAPIC));

      /*
       * The LVT thermal monitor register was introduced
       * in Pentium 4 and Xeon processors.
       */

      if (APIC_MAX_LVT(vm->hostAPIC) >= 5) {
         *thermalNMI = DisableNMIDelivery(&APIC_THERM_REG(vm->hostAPIC));
      } else {
         *thermalNMI = FALSE;
      }
   }
}


static void
RestoreNMI(VMDriver *vm,    // IN
           Bool lint0NMI,   // IN
           Bool lint1NMI,   // IN
           Bool pcNMI,      // IN
           Bool thermalNMI) // IN
{
#define RestoreNMIDelivery(cond, apicr)				\
   do {								\
      if (cond) {						\
         uint32 reg;						\
	 							\
         reg = apicr;						\
         apicr = reg & ~APIC_LVT_MASK;				\
      }								\
   } while (0)

   RestoreNMIDelivery(lint0NMI, APIC_LINT0_REG(vm->hostAPIC));
   RestoreNMIDelivery(lint1NMI, APIC_LINT1_REG(vm->hostAPIC));
   RestoreNMIDelivery(pcNMI, APIC_PC_REG(vm->hostAPIC));
   RestoreNMIDelivery(thermalNMI, APIC_THERM_REG(vm->hostAPIC));
#undef RestoreNMIDelivery
}


#ifdef VM_X86_64
/*
 * From an email with Petr regarding gcc's handling of the stdcall
 * attribute for x86-64:
 *
 *    As far as I can tell, for x86_64 there is only one calling
 *    convention:
 *       On Linux rdi/rsi/rdx/rcx/r8d/r9d for <= 6 arguments,
 *       others always on stack, caller always adjusts stack.
 *
 *       On Windows it is rcx/rdx/r8d/r9d for <= 4 arguments, rest on
 *       stack.  When more than 4 arguments are passed, spill space is
 *       reserved on the stack for the register argumnents.  Argument
 *       5 is accessed at (5 * 8)(rsp).
 */

static INLINE_SINGLE_CALLER void
SwitchToMonitor32(VMCrossPage *crosspage)
{
#if defined(__GNUC__)
#define PROTOPARMS32  uint32       zero0   /* rdi */,   \
                      uint32       zero1   /* rsi */,   \
                      uint32       zero2   /* rdx */,   \
                      VMCrossPage *cp      /* rcx */
#define PARMS32 0 /* rdi */, 0 /* rsi */, 0 /* rdx */, crosspage /* rcx */
#elif defined(_MSC_VER)
#define PROTOPARMS32 VMCrossPage *cp /* rcx */
#define PARMS32 crosspage /* rcx */
#else
#error No compiler defined for 64-bit SwitchToMonitor32
#endif
   typedef void (*SwitchFn)(PROTOPARMS32);
   const uint8    *addr        = &crosspage->contextSwitch.hostXToVmm32[0];
   const SwitchFn  worldSwitch = (SwitchFn)addr;
   worldSwitch(PARMS32);
#undef PROTOPARMS32
#undef PARMS32
}

static INLINE_SINGLE_CALLER void
SwitchToMonitor64(VMCrossPage *crosspage)
{
#if defined(__GNUC__)
   typedef void (*SwitchFn)(uint32   zero0  /* rdi */,
                            uint32   zero1  /* rsi */,
                            uint64   src    /* rdx */,
                            uint64   dest   /* rcx */,
                            Selector hostCS /* r8d */);
#define SWITCHFN64(_fn, _src, _dest, _hostCS)           \
               _fn(0       /* rdi */, 0     /* rsi */,  \
                   _src    /* rdx */, _dest /* rcx */,  \
                   _hostCS /* r8d */)
#elif defined(_MSC_VER)
   typedef void (__cdecl *SwitchFn)(uint64   dest   /* rcx */,
                                    uint64   src    /* rdx */,
                                    Selector hostCS /* r8d */);
#define SWITCHFN64(_fn, _src, _dest, _hostCS)           \
               _fn(_dest   /* rcx */, _src /* rdx */,   \
                   _hostCS /* r8d */)
#else
#error No compiler defined for 64-bit SwitchToMonitor64
#endif

   const uint8    *worldSwitch = crosspage->contextSwitch.hostXToVmm64;
   const SwitchFn  fn          = (SwitchFn)worldSwitch;
   const uint64    hostContext = (uint64)&crosspage->hostContext64;
   const uint64    monContext  = (uint64)&crosspage->monContext64;
   const Selector  hostCS      = (Selector)crosspage->hostContext64.context.cs;

   SWITCHFN64(fn, hostContext, monContext, hostCS);
#undef SWITCHFN64
}

#else   /* VM_X86_64 */

/*
 *-----------------------------------------------------------------------------
 *
 * SwitchToMonitor --
 *
 *      Wrapper that calls code to switch from the host to the monitor.
 *
 * Side effects:
 *      None for the module
 *
 *-----------------------------------------------------------------------------
 */
static INLINE_SINGLE_CALLER void
SwitchToMonitor32(VMCrossPage *crosspage)
{
#if defined(__GNUC__)
#define SWITCHFN (__attribute__((stdcall, regparm(0))) *SwitchFn)
#elif defined(_MSC_VER)
#define SWITCHFN (__stdcall *SwitchFn)   
#else
#error No compiler defined for 32-bit SwitchToMonitor32
#endif
   typedef void SWITCHFN(Selector     cs,
                         ContextInfo *src,
                         ContextInfo *dst,
                         uint32       newDstVA);

   ContextInfo    *src     = &(crosspage->hostContext);
   ContextInfo    *dst     = &(crosspage->monContext);
   const char     *codePtr = &crosspage->contextSwitch.hostXToVmm32[0];
   const Selector  hostCS  = crosspage->hostContext.task.cs;
    /*
    * newDstVA and vm->crosspage->monContext point to the same
    * location.  The latter refers to the monitor's virtual address
    * space.
    */
   uint32 newDstVA  = (uint32)(VPN_2_VA(MODULECALL_CROSS_PAGE_START) +
                               offsetof(VMCrossPage, monContext));    
   SwitchFn fn = (SwitchFn)codePtr;
   fn(hostCS, src, dst, newDstVA);
#undef SWITCHFN
}

static INLINE_SINGLE_CALLER void
SwitchToMonitor64(VMCrossPage *crosspage)
{
#if defined(__GNUC__)
#define SWITCHFN (__attribute__((stdcall, regparm(0))) *SwitchFn)
#elif defined(_MSC_VER)
#define SWITCHFN (__stdcall *SwitchFn)   
#else
#error No compiler defined for 32-bit SwitchToMonitor64
#endif
   typedef void SWITCHFN(Selector     cs,
                         VMCrossPage *crosspage);

   const char     *codePtr = crosspage->contextSwitch.hostXToVmm64;
   const Selector  hostCS  = crosspage->hostContext.task.cs;
   SwitchFn fn = (SwitchFn)codePtr;
   fn(hostCS, crosspage);
#undef SWITCHFN
}

#endif // VM_X86_64

static INLINE_SINGLE_CALLER void
SwitchToMonitor(VMCrossPage *crosspage)
{
   if (crosspage->runVmm64) {
      SwitchToMonitor64(crosspage);
   } else {
      SwitchToMonitor32(crosspage);
   }
}


/*
 *-----------------------------------------------------------------------------
 *
 * Task_Switch --
 *
 *      Switches from the host context into the monitor
 *      context. Think of it as a coroutine switch that changes
 *      not only the registers, but also the address space
 *      and all the hardware state.
 *
 * Results:
 *      Next module call (or user call for that matter) is
 *      returned.
 *
 * Side effects:
 *      Jump to the other side. Has no direct effect on the
 *      host-visible state except that it might generate an interrupt.
 *
 *-----------------------------------------------------------------------------
 */

void 
Task_Switch(VMDriver *vm,  // IN
            Vcpuid vcpuid) // IN
{
   uintptr_t   flags, cr0reg, cr2reg, cr4reg, new_cr4;
   uintptr_t   drReg;
#ifdef VM_X86_64
   uint64      kgs64, gs64, fs64;
#endif
   DTR hostGDT;
   Selector    cs, gs, fs;
   Selector    trReg;
   Selector    hostLDT;
   Bool lint0NMI = FALSE;
   Bool lint1NMI = FALSE;
   Bool pcNMI = FALSE;
   Bool thermalNMI = FALSE;
   VMCrossPage *crosspage = vm->crosspage[vcpuid];
   
#if defined(linux) && defined(USE_PERFCTRS_HOSTED)
   PerfCtr_SnapshotFromHost();
#endif
   DisableNMI(vm, &lint0NMI, &lint1NMI, &pcNMI, &thermalNMI);
   SAVE_FLAGS(flags); 
   CLEAR_INTERRUPTS();

   vm->currentHostCpu[vcpuid] = HostIF_GetCurrentPCPU();
   
   /*
    * Save CR state (not CR3!)
    */
   GET_CR0(cr0reg);
   GET_CR2(cr2reg);
   GET_CR4(cr4reg);

   /* Ensure global pages are flushed */
   new_cr4 = cr4reg & ~CR4_PGE;
   SET_CR4(new_cr4);

   crosspage->hostCR4 = new_cr4;

   /*
    * Save DR7 since we need to disable debug breakpoints during the
    * world switch code.  Save DR6 in order to accomodate the ICEBP
    * instruction.  All other debug registers are saved lazily by the
    * monitor prior to first use.  NOTE: Since monitor is in legacy
    * mode save all DRs for x86_64 here.
    */

   crosspage->hostDRSaved = 0;
   crosspage->hostDRInHW = 0xff;
   
#define SAVE_DR(n)                             \
           GET_DR##n(drReg);                   \
           crosspage->hostDR[n] = drReg;       \
           crosspage->hostDRSaved |= (1 << n);

   if (vm_x86_64) {
      SAVE_DR(0);
      SAVE_DR(1);
      SAVE_DR(2);
      SAVE_DR(3);
   }

   SAVE_DR(6);
   SAVE_DR(7);

#undef SAVE_DR

   /*
    * Try to disable debug exceptions during the switch.
    * Unfortunately we cannot do this reliably, as the host
    * may have set DR7_GD.  This will cause the SET_DR7 to
    * trap, and the host trap handler can then put whatever
    * it wants in DR7 and resume after the SET_DR7.  We fix
    * this in the monitor, with our own trap handler.
    */

   if (UNLIKELY(crosspage->hostDR[7] & DR7_ENABLED)) {
      SET_DR7(DR7_DEFAULT);
   }

   GET_GDT(hostGDT);
   GET_LDT(hostLDT); 

   /*
    * GS and FS are saved outside of the SwitchToMonitor() code to 1)
    * minimize the amount of code handled there and 2) prevent us from
    * faulting if they happen to be in the LDT (since the LDT is saved and
    * restored here too) and 3) make sure that if we do fault (because the
    * uncached descriptor for GS or FS's Selector has become invalid) we
    * fault in the host's context rather than the monitor or, worse,
    * never-never land. --Jeremy.
    */
#ifdef VM_X86_64
   kgs64 = GET_KernelGS64();
   gs64 = GET_GS64();
   fs64 = GET_FS64();
#endif
   cs = GET_CS();
   gs = GET_GS();
   fs = GET_FS();
   GET_TR(trReg);
   
   /* Save the %cs and %tr. */
   if (vm_x86_64) {
      crosspage->hostContext64.tr = trReg;
      crosspage->hostContext64.context.cs = cs;
   } else {
      crosspage->hostContext.tr = trReg;
      crosspage->hostContext.task.cs = cs;
   }

   SetupTemporaryGDT(vm->currentHostCpu[vcpuid], crosspage, hostGDT, cs, trReg);

   if (trReg) {
      /* To return to the task, mark it as unused. */
      Descriptor *desc;
      DTR      *hostContextGDT = GetHostContextGDT(crosspage);
      desc = (Descriptor *)(HOST_KERNEL_LA_2_VA(hostContextGDT->offset) + trReg);
      if (Desc_Type(desc) == TASK_DESC_BUSY) {
         Desc_SetType(desc, TASK_DESC);
      }
   }

   SwitchToMonitor(crosspage); /* See comment for SwitchToMonitor() */

   SET_CR0(cr0reg);
   SET_CR2(cr2reg);
   SET_CR4(cr4reg);
   /* The monitor shouldn't modify CR8 */
   
   if (USE_TEMPORARY_GDT) {
      /* When enabled, vmmon is reentered on the temporary GDT. */
      SET_GDT(hostGDT);
   }
   SET_LDT(hostLDT);

   /* restore fs/gs must come before 64 bit fs/gs restore */
   SET_FS(fs);
   SET_GS(gs);
#ifdef VM_X86_64
   SET_FS64(fs64);
   SET_GS64(gs64);
   SET_KernelGS64(kgs64);
#endif

#ifdef VM_X86_64
#define CAST_HOST_DR(x) (x)
#else
#define CAST_HOST_DR(x) ((uint32) x)
#endif

#define RESTORE_DR(n) if ((crosspage->hostDRInHW & (1 << n)) == 0) {     \
                          drReg = CAST_HOST_DR(crosspage->hostDR[n]);    \
                          SET_DR##n(drReg);                              \
                      }

   RESTORE_DR(0);
   RESTORE_DR(1);
   RESTORE_DR(2);
   RESTORE_DR(3);
   RESTORE_DR(6);
   RESTORE_DR(7);

   if (UNLIKELY(crosspage->restoreGeneralDetect)) {
      crosspage->restoreGeneralDetect = 0;
      drReg = CAST_HOST_DR(crosspage->hostDR[7]) | DR7_GD;
      SET_DR7(drReg);
   }

#undef RESTORE_DR
#undef CAST_HOST_DR

   ASSERT_NO_INTERRUPTS();

   if (crosspage->moduleCallType == MODULECALL_INTR) {
      /*
       * Note we must do the RAISE_INTERRUPT before ever enabling
       * interrupts or bad things have happened (might want to know exactly
       * what bad things btw).
       * Note2: RAISE_INTERRUPT() only takes an constant and hence with switch
       * statement.
       */

#define IRQ_INT(_x) case _x: RAISE_INTERRUPT(_x); break 
#define IRQ_INT2(_x) IRQ_INT(_x); IRQ_INT(_x + 1)
#define IRQ_INT4(_x) IRQ_INT2(_x); IRQ_INT2(_x + 2)
#define IRQ_INT8(_x) IRQ_INT4(_x); IRQ_INT4(_x + 4)
#define IRQ_INT16(_x) IRQ_INT8(_x); IRQ_INT8(_x + 8)
#define IRQ_INT32(_x) IRQ_INT16(_x); IRQ_INT16(_x + 16)

      switch (crosspage->args[0]) {
	 // These are the general IO interrupts
	 // It would be nice to generate this dynamically, but see Note2 above.

	 /*
	  * Pass Machine Check Exception (Interrupt 0x12) to the host.
	  * See bug #45286 for details.
	  */
	 IRQ_INT(0x12);

         /*
          * pass the reserved vectors (20-31) as well. amd64 windows
          * generates these.
          */
	 IRQ_INT8(0x14);
	 IRQ_INT4(0x1c);

	 IRQ_INT32(0x20);
	 IRQ_INT32(0x40);
	 IRQ_INT32(0x60);
	 IRQ_INT32(0x80);
	 IRQ_INT32(0xa0);
	 IRQ_INT32(0xc0);
	 IRQ_INT32(0xe0);

      default: 
	 /*
	  * XXXX nt
	  * running on a 2 processor machine we hit this Panic with int 0xD1 0x61 ...
	  */
	 Warning("Received Unexpected Interrupt: 0x%X in Task_Switch()\n", crosspage->args[0]);
	 Panic("Received Unexpected Interrupt: 0x%X\n", crosspage->args[0]);
      }
   }
   
   vm->currentHostCpu[vcpuid] = INVALID_HOST_CPU;

   RESTORE_FLAGS(flags);
   RestoreNMI(vm, lint0NMI, lint1NMI, pcNMI, thermalNMI);
#if defined(linux) && defined(USE_PERFCTRS_HOSTED)
   PerfCtr_SnapshotFromMonitor();
#endif
}
