/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "libgadget-view-host.h"

#include <glib.h>
#include <string>

#include <ggadget/decorated_view_host.h>
#include <ggadget/floating_main_view_decorator.h>
#include <ggadget/gadget.h>
#include <ggadget/gadget_consts.h>

#include "cairo_canvas.h"
#include <ggadget/gtk/menu_builder.h>
#include <ggadget/gtk/utilities.h>

#include <libgadget/libgadget.h>

using namespace ggadget::gtk;

namespace ggadget {


LibgadgetViewHost::LibgadgetViewHost (GGadget *ggadget,
                                      ViewHostInterface::Type type)
  : ggadget_(ggadget),
    view_(NULL),
    type_(type),
    zoom_(1.0),
    context_menu_(NULL)
{
  
}

LibgadgetViewHost::~LibgadgetViewHost ()
{
  if (GTK_IS_WIDGET (context_menu_))
    gtk_widget_destroy (context_menu_);
}

void
LibgadgetViewHost::Destroy ()
{
  delete this;
}

ViewHostInterface::Type
LibgadgetViewHost::GetType () const
{
  return type_;
}

void
LibgadgetViewHost::SetView (ViewInterface *view)
{
  if (view_ == view)
    return;

  view_ = view;
}

ViewInterface *
LibgadgetViewHost::GetView () const
{
  return view_;
}

void
LibgadgetViewHost::Draw (cairo_t *cr)
{
  CairoCanvas *canvas = new CairoCanvas (cr,
                                         1.0,
                                         view_->GetWidth(),
                                         view_->GetHeight());
  view_->Draw (canvas);
  canvas->Destroy ();
}

void *
LibgadgetViewHost::GetNativeWidget () const
{
  return NULL;
}

void
LibgadgetViewHost::ViewCoordToNativeWidgetCoord (double x,
                                                 double y,
                                                 double *widget_x,
                                                 double *widget_y) const
{
  if (widget_x)
    *widget_x = x * zoom_;

  if (widget_y)
    *widget_y = y * zoom_;
}

void
LibgadgetViewHost::NativeWidgetCoordToViewCoord (double x,
                                                 double y,
                                                 double *view_x,
                                                 double *view_y) const
{
  if (view_x)
    *view_x = x/zoom_;
  if (view_y)
    *view_y = y/zoom_;
}

void
LibgadgetViewHost::QueueDraw ()
{
  g_signal_emit_by_name (ggadget_, "queue-redraw");
}

void
LibgadgetViewHost::QueueResize ()
{
  gint width;
  gint height;

  width = (gint)view_->GetWidth ();
  height = (gint)view_->GetHeight ();

  /* Unless the canvas is even, text doesn't render properly...go figure */
  if (width % 2)
    width += 1;
  if (height % 2)
    height += 1;
  gadget_proxy_set_width (GADGET_PROXY (ggadget_), width);
  gadget_proxy_set_height (GADGET_PROXY (ggadget_), height);
  g_signal_emit_by_name (ggadget_, "queue-resize");
}

void
LibgadgetViewHost::EnableInputShapeMask (bool enable)
{
  QueueDraw ();
}

void
LibgadgetViewHost::SetResizable (ViewInterface::ResizableMode mode)
{
  //debug ("SetResizable");
}

void
LibgadgetViewHost::SetCaption (const std::string &caption)
{
  //g_debug ("SetCaption: %s", caption.c_str ());
}

void
LibgadgetViewHost::SetShowCaptionAlways (bool always)
{
  ;
}

void
LibgadgetViewHost::SetCursor (ggadget::ViewInterface::CursorType cursor)
{
  //g_debug ("SetCursor");
}

void
LibgadgetViewHost::ShowTooltip (const std::string &tooltip)
{
  //g_debug ("ShowToolip: %s", tooltip.c_str ());
}

void
LibgadgetViewHost::ShowTooltipAtPosition (const std::string &tooltip,
                                        double             x,
                                        double             y)
{
  //g_debug ("ShowTooltipAtPosition: %s %.2f %.2f", tooltip.c_str (), x, y);
}

bool
LibgadgetViewHost::ShowView (bool modal, 
                           int flags, 
                           Slot1<bool, int> *feedback_handler)
{
  view_->OnOtherEvent (SimpleEvent (Event::EVENT_OPEN));
  return true;
}

void
LibgadgetViewHost::CloseView ()
{
  g_debug ("CloseView");
}

bool
LibgadgetViewHost::ShowContextMenu (int button)
{
  if (context_menu_)
    gtk_widget_destroy(context_menu_);

  context_menu_ = gtk_menu_new();
  MenuBuilder menu_builder (GTK_MENU_SHELL (context_menu_));

  view_->OnAddContextMenuItems (&menu_builder);

  if (menu_builder.ItemAdded()) 
  {
    int gtk_button;
    switch (button) 
    {
      case MouseEvent::BUTTON_LEFT: gtk_button = 1; break;
      case MouseEvent::BUTTON_MIDDLE: gtk_button = 2; break;
      case MouseEvent::BUTTON_RIGHT: gtk_button = 3; break;
        default: gtk_button = 3; break;
    }

    gtk_menu_popup (GTK_MENU (context_menu_),
                    NULL, NULL, NULL, NULL,
                    gtk_button, gtk_get_current_event_time());
    return true;
  }
  return false;
}

void
LibgadgetViewHost::Alert (const ViewInterface *view,
                          const char *message)
{
  ShowAlertDialog ("Google Gadgets", message);
}

ggadget::ViewHostInterface::ConfirmResponse
LibgadgetViewHost::Confirm (const ViewInterface *view,
                            const char *message,
                            bool cancel_button)
{
  return ShowConfirmDialog (view->GetCaption().c_str(), message, cancel_button);
}

std::string
LibgadgetViewHost::Prompt (const ViewInterface *view,
                           const char *message,
                           const char *default_value)
{
  return ShowPromptDialog ("Google Gadgets",
                           message,
                           default_value);
}

int
LibgadgetViewHost::GetDebugMode () const
{
  return 3;
}

void
LibgadgetViewHost::BeginResizeDrag (int, ViewInterface::HitTest)
{
  GadgetResizePoint point = GADGET_RESIZE_BOTTOMRIGHT;

  g_signal_emit_by_name (ggadget_, "begin-resize", point);
}

void
LibgadgetViewHost::BeginMoveDrag (int)
{
  g_signal_emit_by_name (ggadget_, "begin-drag-move");
}

} /* namespace ggadget */
