/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <string.h>
#include <arpa/inet.h>
#include <nm-utils.h>
#include "nmn-connection-details.h"

G_DEFINE_TYPE (NmnConnectionDetails, nmn_connection_details, GTK_TYPE_TABLE)

enum {
    PROP_0,
    PROP_EDITABLE,

    LAST_PROP
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_CONNECTION_DETAILS, NmnConnectionDetailsPrivate))

typedef struct {
    GtkComboBox *method_combo;
    GtkWidget *address;
    GtkWidget *netmask;
    GtkWidget *gateway;
    GtkWidget *dns;
    gboolean editable;

    gboolean disposed;
} NmnConnectionDetailsPrivate;

#define METHOD_AUTO       0
#define METHOD_MANUAL     1
#define METHOD_LINK_LOCAL 2

GtkWidget *
nmn_connection_details_new (gboolean editable)
{
    return GTK_WIDGET (g_object_new (NMN_TYPE_CONNECTION_DETAILS,
                                     NMN_CONNECTION_DETAILS_EDITABLE, editable,
                                     NULL));
}

void
nmn_connection_details_set_setting (NmnConnectionDetails *self,
                                    NMSettingIP4Config *setting)
{
    NmnConnectionDetailsPrivate *priv;
    const char *method;
    int active_method;

    g_return_if_fail (NMN_IS_CONNECTION_DETAILS (self));
    g_return_if_fail (NM_IS_SETTING_IP4_CONFIG (setting));

    priv = GET_PRIVATE (self);

    method = nm_setting_ip4_config_get_method (setting);
    if (!strcmp (method, NM_SETTING_IP4_CONFIG_METHOD_AUTO))
        active_method = METHOD_AUTO;
    else if (!strcmp (method, NM_SETTING_IP4_CONFIG_METHOD_MANUAL))
        active_method = METHOD_MANUAL;
    else if (!strcmp (method, NM_SETTING_IP4_CONFIG_METHOD_LINK_LOCAL))
        active_method = METHOD_LINK_LOCAL;
    else
        active_method = -1;

    gtk_combo_box_set_active (priv->method_combo, active_method);
}

static char *
ip4_address_as_string (guint32 ip)
{
    char *ip_string;
    struct in_addr tmp_addr;

    tmp_addr.s_addr = ip;
    ip_string = g_malloc0 (INET_ADDRSTRLEN + 1);
    if (!inet_ntop (AF_INET, &tmp_addr, ip_string, INET_ADDRSTRLEN))
        strcpy (ip_string, "(none)");

    return ip_string;
}

static inline GtkWidget *
aligned_label_new (const char *text)
{
    GtkWidget *w;

    w = gtk_label_new (text);
    g_object_set (w, "xalign", 0.0, NULL);

    return w;
}

static GtkWidget *
create_text_widget (GtkWidget *current_widget,
                    GtkTable *table,
                    int col,
                    int row,
                    gboolean editable)
{
    GtkWidget *w;
    GType target_type;

    target_type = editable ? GTK_TYPE_ENTRY : GTK_TYPE_LABEL;

    if (current_widget && G_TYPE_CHECK_INSTANCE_TYPE (current_widget, target_type))
        /* all well, nothing to do */
        return current_widget;

    if (current_widget)
        gtk_widget_destroy (current_widget);

    if (editable)
        w = gtk_entry_new ();
    else
        w = aligned_label_new ("");

    gtk_widget_show (w);
    gtk_table_attach_defaults (table, w, col, col + 1, row, row + 1);

    return w;
}

static void
editable_changed (NmnConnectionDetails *self)
{
    NmnConnectionDetailsPrivate *priv = GET_PRIVATE (self);
    GtkTable *table = GTK_TABLE (self);

    priv->editable = gtk_combo_box_get_active (priv->method_combo) == METHOD_MANUAL;

    priv->address = create_text_widget (priv->address, table, 1, 1, priv->editable);
    priv->netmask = create_text_widget (priv->netmask, table, 1, 2, priv->editable);
    priv->gateway = create_text_widget (priv->gateway, table, 1, 3, priv->editable);

    gtk_text_view_set_editable (GTK_TEXT_VIEW (priv->dns), priv->editable);
}

static inline void
widget_set_text (GtkWidget *w, char *str)
{
    if (GTK_IS_LABEL (w))
        gtk_label_set_text (GTK_LABEL (w), str);
    else if (GTK_IS_ENTRY (w))
        gtk_entry_set_text (GTK_ENTRY (w), str);
    else
        g_warning ("Invalid widget type '%s'", G_OBJECT_TYPE_NAME (w));

    g_free (str);
}

void
nmn_connection_details_set_config (NmnConnectionDetails *self,
                                   NMIP4Config *config)
{
    NmnConnectionDetailsPrivate *priv;
    const GSList *list;
    const GArray *array;
    char *str;
    GtkTextBuffer *buffer;

    g_return_if_fail (NMN_IS_CONNECTION_DETAILS (self));
    g_return_if_fail (NM_IS_IP4_CONFIG (config));

    priv = GET_PRIVATE (self);

    list = nm_ip4_config_get_addresses (config);
    if (list) {
        NMIP4Address *def_addr = (NMIP4Address *) list->data;
        guint32 netmask;

        str = ip4_address_as_string (nm_ip4_address_get_address (def_addr));
        widget_set_text (priv->address, str);

        netmask = nm_utils_ip4_prefix_to_netmask (nm_ip4_address_get_prefix (def_addr));
        str = ip4_address_as_string (netmask);
        widget_set_text (priv->netmask, str);

        if (nm_ip4_address_get_gateway (def_addr)) {
            str = ip4_address_as_string (nm_ip4_address_get_gateway (def_addr));
            widget_set_text (priv->gateway, str);
        }
    }

    array = nm_ip4_config_get_nameservers (config);
	if (array) {
        GString *string;
        int i;

        string = g_string_sized_new (50);
        for (i = 0; i < array->len; i++) {
            if (i > 0)
                g_string_append_c (string, '\n');

            str = ip4_address_as_string (g_array_index (array, guint32, i));
            g_string_append (string, str);
            g_free (str);
        }

        str = g_string_free (string, FALSE);
	} else
        str = NULL;

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->dns));
    gtk_text_buffer_set_text (buffer, str, -1);
    g_free (str);
}

static void
nmn_connection_details_init (NmnConnectionDetails *details)
{
    NmnConnectionDetailsPrivate *priv = GET_PRIVATE (details);
    GtkTable *table;
    GtkWidget *w;

    g_object_set (details,
                  "n-rows", 5,
                  "n-columns", 2,
                  "homogeneous", FALSE,
                  "row-spacing", 6,
                  "column-spacing", 6,
                  NULL);

    table = GTK_TABLE (details);

    w = aligned_label_new ("Connect by:");
    gtk_table_attach_defaults (table, w, 0, 1, 0, 1);

    w = aligned_label_new ("IP Address:");
    gtk_table_attach_defaults (table, w, 0, 1, 1, 2);

    w = aligned_label_new ("Subnet mask:");
    gtk_table_attach_defaults (table, w, 0, 1, 2, 3);

    w = aligned_label_new ("Router:");
    gtk_table_attach_defaults (table, w, 0, 1, 3, 4);

    w = aligned_label_new ("DNS:");
    gtk_table_attach_defaults (table, w, 0, 1, 4, 5);

    priv->dns = gtk_text_view_new ();
    gtk_table_attach_defaults (table, priv->dns, 1, 2, 4, 5);

    priv->method_combo = GTK_COMBO_BOX (gtk_combo_box_new_text ());
    gtk_combo_box_append_text (priv->method_combo, "DHCP");
    gtk_combo_box_append_text (priv->method_combo, "Manual");
    gtk_combo_box_append_text (priv->method_combo, "Link Local");
    gtk_table_attach_defaults (table, GTK_WIDGET (priv->method_combo), 1, 2, 0, 1);
    g_signal_connect_swapped (priv->method_combo, "changed", G_CALLBACK (editable_changed), details);
    gtk_combo_box_set_active (priv->method_combo, METHOD_AUTO);
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnConnectionDetailsPrivate *priv;

    object = G_OBJECT_CLASS (nmn_connection_details_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    gtk_widget_set_sensitive (GTK_WIDGET (priv->method_combo), priv->editable);
    gtk_widget_show_all (GTK_WIDGET (object));

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnConnectionDetailsPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_EDITABLE:
        priv->editable = g_value_get_boolean (value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnConnectionDetailsPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_EDITABLE:
        g_value_set_boolean (value, priv->editable);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnConnectionDetailsPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_connection_details_parent_class)->dispose (object);
}

static void
nmn_connection_details_class_init (NmnConnectionDetailsClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnConnectionDetailsPrivate));

    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_EDITABLE,
         g_param_spec_boolean (NMN_CONNECTION_DETAILS_EDITABLE,
                               "Editable",
                               "Editable",
                               FALSE,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

}
