/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <string.h>
#include <nm-settings.h>
#include <nm-setting-connection.h>
#include <nm-setting-wireless.h>
#include "nmn-wifi-handler.h"
#include "nmn-wifi-item.h"
#include "utils.h"

G_DEFINE_TYPE (NmnWifiHandler, nmn_wifi_handler, NMN_TYPE_DEVICE_HANDLER)

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_WIFI_HANDLER, NmnWifiHandlerPrivate))

typedef struct {
    gulong wifi_toggled_id;
    gulong ap_added_id;
    gulong ap_removed_id;
    gulong ap_changed_id;

    gboolean disposed;
} NmnWifiHandlerPrivate;

NmnDeviceHandler *
nmn_wifi_handler_new (NmnNMData *nm_data,
                      NMDeviceWifi *device)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);
    g_return_val_if_fail (NM_IS_DEVICE_WIFI (device), NULL);

    return NMN_DEVICE_HANDLER (g_object_new (NMN_TYPE_WIFI_HANDLER,
                                             NMN_DEVICE_HANDLER_NM_DATA, nm_data,
                                             NMN_DEVICE_HANDLER_DEVICE, device,
                                             NULL));
}

static NMAccessPoint *
find_best_ap_for_connection (NMDeviceWifi *device,
                             NMExportedConnection *connection)
{
    const GPtrArray *aps;
    NMConnection *wrapped;
    NMAccessPoint *best_ap = NULL;
    int i;

    wrapped = nm_exported_connection_get_connection (connection);
    aps = nm_device_wifi_get_access_points (device);
    for (i = 0; aps && i < aps->len; i++) {
        NMAccessPoint *ap = NM_ACCESS_POINT (g_ptr_array_index (aps, i));

        if (utils_connection_valid_for_device (wrapped, NM_DEVICE (device), ap)) {
            if (!best_ap)
                best_ap = ap;
            else if (nm_access_point_get_strength (best_ap) < nm_access_point_get_strength (ap))
                best_ap = ap;
        }
    }

    return best_ap;
}

static void
update_items (NmnWifiHandler *self)
{
    GSList *list;
    GSList *iter;
    NMDeviceWifi *device;

    device = NM_DEVICE_WIFI (nmn_device_handler_get_device (NMN_DEVICE_HANDLER (self)));
    list = nmn_device_handler_get_items (NMN_DEVICE_HANDLER (self));

    for (iter = list; iter; iter = iter->next) {
        NmnWifiItem *item = NMN_WIFI_ITEM (iter->data);
        NMAccessPoint *current_ap = nmn_wifi_item_get_ap (item);
        NMAccessPoint *best_ap;

        best_ap = nm_device_wifi_get_active_access_point (device);
        if (!best_ap) {
            NMExportedConnection *exported;

            exported = nmn_network_item_get_connection (NMN_NETWORK_ITEM (item));
            best_ap = find_best_ap_for_connection (device, exported);
        }

        if (!best_ap)
            nmn_item_remove_request (NMN_ITEM (item));
        else if (best_ap != current_ap)
            nmn_wifi_item_set_ap (item, best_ap);
    }
}

static void
ap_updated (NMAccessPoint *ap,
            GParamSpec *pspec,
            gpointer user_data)
{
    update_items (NMN_WIFI_HANDLER (user_data));
}

static void
ap_removed (NMDeviceWifi *device,
            NMAccessPoint *ap,
            gpointer user_data)
{
    g_signal_handlers_disconnect_by_func (ap, "notify", ap_updated);
    update_items (NMN_WIFI_HANDLER (user_data));
}

static void
connection_added (NmnDeviceHandler *handler,
                  NMExportedConnection *exported)
{
    NMConnection *wrapped;
    NMSettingConnection *s_con;
    const char *connection_type;
    NMDeviceWifi *device;
    NMAccessPoint *ap;
    GtkWidget *item;

    wrapped = nm_exported_connection_get_connection (exported);
    s_con = NM_SETTING_CONNECTION (nm_connection_get_setting (wrapped, NM_TYPE_SETTING_CONNECTION));
    connection_type = nm_setting_connection_get_connection_type (s_con);

    if (!connection_type || strcmp (connection_type, NM_SETTING_WIRELESS_SETTING_NAME))
        /* Not a wifi connection */
        return;

    /* Make sure it doesn't exist yet */
    if (nmn_device_handler_get_item_for_connection (handler, exported))
        return;

    device = NM_DEVICE_WIFI (nmn_device_handler_get_device (handler));
    ap = find_best_ap_for_connection (device, exported);
    if (ap) {
        item = nmn_wifi_item_new (nmn_device_handler_get_nm_data (handler), device, ap);
        g_object_set (item, NMN_NETWORK_ITEM_CONNECTION, exported, NULL);
        nmn_item_set_delete_visible (NMN_ITEM (item), TRUE);
        nmn_device_handler_add_item (handler, NMN_ITEM (item));
    }
}

static void
ap_added (NMDeviceWifi *device,
          NMAccessPoint *ap,
          gpointer user_data)
{
    NmnDeviceHandler *handler = NMN_DEVICE_HANDLER (user_data);
    GSList *list;
    GSList *iter;

    /* Catch the signals of the new AP */
    g_signal_connect (ap, "notify", G_CALLBACK (ap_updated), handler);

    /* */
    update_items (NMN_WIFI_HANDLER (handler));

    /* Maybe there's an existing connection for it which hasn't been added yet? */
    list = nmn_device_handler_get_connections (handler);
    for (iter = list; iter; iter = iter->next)
        connection_added (handler, NM_EXPORTED_CONNECTION (iter->data));

    g_slist_free (list);
}

static void
active_ap_changed (NMDeviceWifi *device,
                   GParamSpec *pspec,
                   gpointer user_data)
{
    update_items (NMN_WIFI_HANDLER (user_data));
}

static void
wifi_toggled (NmnNMData *nm_data,
              gboolean active,
              gpointer user_data)
{
    NmnDeviceHandler *handler = NMN_DEVICE_HANDLER (user_data);

    if (active)
        nmn_device_handler_add_items (handler);
    else
        nmn_device_handler_remove_items (handler);
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnNMData *nm_data;
    NmnWifiHandlerPrivate *priv;
    NMDeviceWifi *device;
    const GPtrArray *aps;
    int i;

    object = G_OBJECT_CLASS (nmn_wifi_handler_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    nm_data = nmn_device_handler_get_nm_data (NMN_DEVICE_HANDLER (object));
    priv->wifi_toggled_id = g_signal_connect (nm_data, "wifi-toggled", G_CALLBACK (wifi_toggled), object);

    device = NM_DEVICE_WIFI (nmn_device_handler_get_device (NMN_DEVICE_HANDLER (object)));

    priv->ap_added_id   = g_signal_connect (device, "access-point-added", G_CALLBACK (ap_added), object);
    priv->ap_removed_id = g_signal_connect (device, "access-point-removed", G_CALLBACK (ap_removed), object);
    priv->ap_changed_id = g_signal_connect (device, "notify::" NM_DEVICE_WIFI_ACTIVE_ACCESS_POINT,
                                            G_CALLBACK (active_ap_changed), object);

    aps = nm_device_wifi_get_access_points (device);
    for (i = 0; aps && i < aps->len; i++) {
        NMAccessPoint *ap = NM_ACCESS_POINT (g_ptr_array_index (aps, i));

        g_signal_connect (ap, "notify", G_CALLBACK (ap_updated), object);
    }

    return object;
}

static void
nmn_wifi_handler_init (NmnWifiHandler *handler)
{
}

static void
dispose (GObject *object)
{
    NmnWifiHandlerPrivate *priv = GET_PRIVATE (object);
    NmnNMData *nm_data;
    NMDeviceWifi *device;
    const GPtrArray *aps;
    int i;

    if (priv->disposed)
        return;

    nm_data = nmn_device_handler_get_nm_data (NMN_DEVICE_HANDLER (object));
    g_signal_handler_disconnect (nm_data, priv->wifi_toggled_id);

    device = NM_DEVICE_WIFI (nmn_device_handler_get_device (NMN_DEVICE_HANDLER (object)));

    aps = nm_device_wifi_get_access_points (device);
    for (i = 0; aps && i < aps->len; i++) {
        NMAccessPoint *ap = NM_ACCESS_POINT (g_ptr_array_index (aps, i));

        g_signal_handlers_disconnect_by_func (ap, "notify", ap_updated);
    }

    g_signal_handler_disconnect (device, priv->ap_added_id);
    g_signal_handler_disconnect (device, priv->ap_removed_id);
    g_signal_handler_disconnect (device, priv->ap_changed_id);

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_wifi_handler_parent_class)->dispose (object);
}

static void
nmn_wifi_handler_class_init (NmnWifiHandlerClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    NmnDeviceHandlerClass *handler_class = NMN_DEVICE_HANDLER_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnWifiHandlerPrivate));

    object_class->constructor = constructor;
    object_class->dispose = dispose;

    handler_class->connection_added = connection_added;
}
