/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <string.h>
#include <nm-device-wifi.h>
#include <nm-utils.h>
#include "nmn-wifi-list.h"
#include "nmn-wifi-item.h"
#include "utils.h"

G_DEFINE_TYPE (NmnWifiList, nmn_wifi_list, NMN_TYPE_LIST)

enum {
    PROP_0,
    PROP_NM_DATA,

    LAST_PROP
};

enum {
    CONNECT_REQUESTED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL];

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_WIFI_LIST, NmnWifiListPrivate))

typedef struct {
    NmnNMData *nm_data;

    gboolean disposed;
} NmnWifiListPrivate;

GtkWidget *
nmn_wifi_list_new (NmnNMData *nm_data)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);

    return GTK_WIDGET (g_object_new (NMN_TYPE_WIFI_LIST,
                                     NMN_WIFI_LIST_NM_DATA, nm_data,
                                     NULL));
}

static void
connect_requested (NmnItem *item, gpointer user_data)
{
    NmnWifiList *list = NMN_WIFI_LIST (user_data);

    g_signal_emit (list, signals[CONNECT_REQUESTED], 0);
}

static gboolean
matching_connection_exists (NmnWifiList *list, NMDevice *device, NMAccessPoint *ap)
{
    NmnWifiListPrivate *priv = GET_PRIVATE (list);
    GSList *connections;
    GSList *iter;
    gboolean exists = FALSE;

    connections = nm_settings_list_connections (nmn_nm_data_get_user_settings (priv->nm_data));
    for (iter = connections; iter; iter = iter->next) {
        NMConnection *connection = nm_exported_connection_get_connection (NM_EXPORTED_CONNECTION (iter->data));

        if (utils_connection_valid_for_device (connection, device, ap)) {
            exists = TRUE;
            break;
        }
    }

    g_slist_free (connections);

    return exists;
}

typedef struct {
    const GByteArray *ssid;
    NM80211Mode mode;
    guint32 flags;
    guint32 wpa_flags;
    guint32 rsn_flags;

    gboolean have_match;
} MatchingAPInfo;

static void
matching_ap_cb (GtkWidget *widget,
                gpointer user_data)
{
    NmnWifiItem *item = NMN_WIFI_ITEM (widget);
    MatchingAPInfo *info = (MatchingAPInfo *) user_data;
    NMAccessPoint *ap;
    const GByteArray *ssid;

    if (info->have_match)
        return;

    ap = nmn_wifi_item_get_ap (item);

    ssid = nm_access_point_get_ssid (ap);
    if (!ssid || ssid->len != info->ssid->len || memcmp (ssid->data, info->ssid->data, ssid->len))
        return;

    if (nm_access_point_get_mode (ap) == info->mode &&
        nm_access_point_get_flags (ap) == info->flags &&
        nm_access_point_get_wpa_flags (ap) == info->wpa_flags &&
        nm_access_point_get_rsn_flags (ap) == info->rsn_flags)

        info->have_match = TRUE;
}

static gboolean
matching_ap_exists (NmnWifiList *list,
                    NMAccessPoint *ap)
{
    MatchingAPInfo info;

    info.ssid = nm_access_point_get_ssid (ap);
    info.mode = nm_access_point_get_mode (ap);
	info.flags = nm_access_point_get_flags (ap);
	info.wpa_flags = nm_access_point_get_wpa_flags (ap);
	info.rsn_flags = nm_access_point_get_rsn_flags (ap);
    info.have_match = FALSE;

    gtk_container_foreach (GTK_CONTAINER (list), matching_ap_cb, &info);

    return info.have_match;
}

static gboolean
ignore_ap (NmnWifiList *list,
           NMDeviceWifi *device,
           NMAccessPoint *ap)
{
    const GByteArray *ssid;

	/* Don't add BSSs that hide their SSID */
	ssid = nm_access_point_get_ssid (ap);
	if (!ssid || nm_utils_is_empty_ssid (ssid->data, ssid->len))
		return TRUE;

    /* Ignore APs for which a matching connection already exists */
    if (matching_connection_exists (list, NM_DEVICE (device), ap))
        return TRUE;

    /* Filter out APs which would look as duplicates */
    if (matching_ap_exists (list, ap))
        return TRUE;

    return FALSE;
}

static void
ap_added (NMDeviceWifi *device,
          NMAccessPoint *ap,
          gpointer user_data)
{
    NmnWifiList *list = NMN_WIFI_LIST (user_data);
    NmnWifiListPrivate *priv = GET_PRIVATE (list);
    GtkWidget *item;

    if (ignore_ap (list, device, ap))
        return;

    item = nmn_wifi_item_new (priv->nm_data, device, ap);
    nmn_list_add_item (NMN_LIST (list), NMN_ITEM (item));
    g_signal_connect (item, "connect-requested", G_CALLBACK (connect_requested), list);
}

static void
ap_removed (NMDeviceWifi *device,
            NMAccessPoint *ap,
            gpointer user_data)
{
    /* FIXME */

    /*     NmnWifiList *list = NMN_WIFI_LIST (user_data); */
    /*     NmnWifiListPrivate *priv = GET_PRIVATE (list); */
}

static void
device_added (NMClient *client,
              NMDevice *device,
              gpointer user_data)
{
    NMDeviceWifi *wifi_device;
    const GPtrArray *aps;
    int i;

    if (!NM_IS_DEVICE_WIFI (device))
        return;

    /* FIXME: Need to remove the signal handlers on dispose */
    g_signal_connect (device, "access-point-added", G_CALLBACK (ap_added), user_data);
    g_signal_connect (device, "access-point-removed", G_CALLBACK (ap_removed), user_data);

    wifi_device = NM_DEVICE_WIFI (device);
    aps = nm_device_wifi_get_access_points (wifi_device);
    for (i = 0; aps && i < aps->len; i++)
        ap_added (wifi_device, NM_ACCESS_POINT (g_ptr_array_index (aps, i)), user_data);
}

static void
device_removed (NMClient *client,
                NMDevice *device,
                gpointer user_data)
{
    /* FIXME */
    g_debug (G_STRLOC);
}

static gboolean
populate_list (gpointer data)
{
    NmnWifiList *list = NMN_WIFI_LIST (data);
    NmnWifiListPrivate *priv = GET_PRIVATE (list);
    NMClient *client;
    const GPtrArray *devices;
    int i;

    client = NM_CLIENT (priv->nm_data);
    devices = nm_client_get_devices (client);
    for (i = 0; devices && i < devices->len; i++)
        device_added (client, NM_DEVICE (g_ptr_array_index (devices, i)), list);

    return FALSE;
}

static void
nmn_wifi_list_init (NmnWifiList *list)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnWifiListPrivate *priv;
    GSource *source;

    object = G_OBJECT_CLASS (nmn_wifi_list_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    if (!priv->nm_data) {
        g_warning ("Missing constructor arguments");
        g_object_unref (object);
        return NULL;
    }

    /* FIXME: Need to remove the signal handlers on dispose */
    g_signal_connect (priv->nm_data,
                      "device-added",
                      G_CALLBACK (device_added),
                      object);

    g_signal_connect (priv->nm_data,
                      "device-removed",
                      G_CALLBACK (device_removed),
                      object);

    source = g_idle_source_new ();
    g_source_set_closure (source, g_cclosure_new_object (G_CALLBACK (populate_list), object));
    g_source_attach (source, NULL);
    g_source_unref (source);

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnWifiListPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        /* Construct only */
        priv->nm_data = g_value_dup_object (value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnWifiListPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        g_value_set_object (value, priv->nm_data);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnWifiListPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    g_object_unref (priv->nm_data);
    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_wifi_list_parent_class)->dispose (object);
}

static void
nmn_wifi_list_class_init (NmnWifiListClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnWifiListPrivate));

    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_NM_DATA,
         g_param_spec_object (NMN_WIFI_LIST_NM_DATA,
                              "NmnNMData",
                              "NmnNMData",
                              NMN_TYPE_NM_DATA,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    /* Signals */
    signals[CONNECT_REQUESTED] = g_signal_new 
        ("connect-requested",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnWifiListClass, connect_requested),
         NULL, NULL,
         g_cclosure_marshal_VOID__VOID,
         G_TYPE_NONE, 0);
}
