/*
 *
 * Authored By Matthew Allum  <mallum@openedhand.com>
 *
 * Copyright (C) 2007 OpenedHand
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * SECTION:tidy-proxy-texture
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <clutter/clutter-scriptable.h>
#include <clutter/cogl.h>

#include "tidy-proxy-texture.h"
#include "tidy-texture-cache.h"


enum
{
  PROP_0,

  PROP_PATH
};

static void clutter_scriptable_iface_init (ClutterScriptableIface *iface);

G_DEFINE_TYPE_WITH_CODE (TidyProxyTexture,
                         tidy_proxy_texture,
                         CLUTTER_TYPE_ACTOR,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_SCRIPTABLE,
                                                clutter_scriptable_iface_init));

#define TIDY_PROXY_TEXTURE_GET_PRIVATE(obj) \
(G_TYPE_INSTANCE_GET_PRIVATE ((obj), TIDY_TYPE_PROXY_TEXTURE, TidyProxyTexturePrivate))

struct _TidyProxyTexturePrivate
{
  ClutterActor        *real_texture;
  gchar               *path;
};


static void
clone_texture_render_to_gl_quad (TidyProxyTexture    *proxy,
				 int                  x_1, 
				 int                  y_1, 
				 int                  x_2, 
				 int                  y_2)
{
  gint   qx1 = 0, qx2 = 0, qy1 = 0, qy2 = 0;
  gint   qwidth = 0, qheight = 0;
  gint   x, y, i = 0, lastx = 0, lasty = 0;
  gint   n_x_tiles, n_y_tiles; 
  gint   pwidth, pheight;
  float tx, ty;

  TidyProxyTexturePrivate *priv = proxy->priv;  
  ClutterTexture *tex;
  ClutterActor *parent_actor = CLUTTER_ACTOR (priv->real_texture);

  qwidth  = x_2 - x_1;
  qheight = y_2 - y_1;

  if (!CLUTTER_ACTOR_IS_REALIZED (parent_actor))
      clutter_actor_realize (parent_actor);

  tex = clutter_clone_texture_get_parent_texture 
    (CLUTTER_CLONE_TEXTURE(priv->real_texture));

  /* Only paint if parent is in a state to do so */
  if (!clutter_texture_has_generated_tiles (tex))
    return;
  
  clutter_texture_get_base_size (tex, &pwidth, &pheight); 

  if (!clutter_texture_is_tiled (tex))
    {
      clutter_texture_bind_tile (tex, 0);

      /* NPOTS textures *always* used if extension available
       */
      if (clutter_feature_available (CLUTTER_FEATURE_TEXTURE_RECTANGLE))
	{
	  tx = (float) pwidth;
	  ty = (float) pheight;
	}
      else
	{
	  tx = (float) pwidth / clutter_util_next_p2 (pwidth);  
	  ty = (float) pheight / clutter_util_next_p2 (pheight);
	}

      cogl_texture_quad (x_1, x_2, y_1, y_2, 
			 0,
			 0,
			 CLUTTER_FLOAT_TO_FIXED (tx),
			 CLUTTER_FLOAT_TO_FIXED (ty));
      return;
    }

  clutter_texture_get_n_tiles (tex, 
			       &n_x_tiles, &n_y_tiles); 

  for (x = 0; x < n_x_tiles; x++)
    {
      lasty = 0;

      for (y = 0; y < n_y_tiles; y++)
	{
	  gint actual_w, actual_h;
	  gint xpos, ypos, xsize, ysize, ywaste, xwaste;
	  
	  clutter_texture_bind_tile (tex, i);
	 
	  clutter_texture_get_x_tile_detail 
	                  (tex, 
			   x, &xpos, &xsize, &xwaste);

	  clutter_texture_get_y_tile_detail 
	                  (tex, 
			   y, &ypos, &ysize, &ywaste);

	  actual_w = xsize - xwaste;
	  actual_h = ysize - ywaste;

	  tx = (float) actual_w / xsize;
	  ty = (float) actual_h / ysize;

	  qx1 = x_1 + lastx;
	  qx2 = qx1 + ((qwidth * actual_w ) / pwidth );
	  
	  qy1 = y_1 + lasty;
	  qy2 = qy1 + ((qheight * actual_h) / pheight );

	  cogl_texture_quad (qx1, qx2, qy1, qy2, 
			     0,
			     0,
			     CLUTTER_FLOAT_TO_FIXED (tx),
			     CLUTTER_FLOAT_TO_FIXED (ty));

	  lasty += qy2 - qy1;	  

	  i++;
	}
      lastx += qx2 - qx1;
    }
}

static void
tidy_proxy_texture_set_path (TidyProxyTexture *self,
                             const gchar      *path)
{
  TidyProxyTexturePrivate *priv = self->priv;
  gchar *real_path;

  if (!path || path[0] == '\0')
    return;

  if (g_path_is_absolute (path))
    real_path = g_strdup (path);
  else
    {
      gchar *dir = g_get_current_dir ();

      real_path = g_build_filename (dir, path, NULL);
      g_free (dir);
    }

  if (priv->path && (strcmp (priv->path, real_path) == 0))
    {
      g_free (real_path);
      return;
    }

  g_free (priv->path);

  priv->path = real_path;
  priv->real_texture =
    tidy_texture_cache_get_texture (tidy_texture_cache_get_default (),
                                    priv->path,
                                    TRUE);

  g_object_notify (G_OBJECT (self), "path");
}

static void
tidy_proxy_texture_paint (ClutterActor *self)
{
  TidyProxyTexture         *proxy = TIDY_PROXY_TEXTURE(self);
  TidyProxyTexturePrivate  *priv = proxy->priv;  

  ClutterActor             *parent_texture;
  gint                      x_1, y_1, x_2, y_2;
  GLenum                    target_type;
  ClutterColor              col = { 0xff, 0xff, 0xff, 0xff };

  if (priv->real_texture == NULL)
    return;

  parent_texture = CLUTTER_ACTOR (priv->real_texture);
  if (!CLUTTER_ACTOR_IS_REALIZED (parent_texture))
    clutter_actor_realize (parent_texture);

  cogl_push_matrix ();

  /* FIXME: figure out nicer way of getting at this info...  
   */  
  if (clutter_feature_available (CLUTTER_FEATURE_TEXTURE_RECTANGLE) &&
      clutter_texture_is_tiled (clutter_clone_texture_get_parent_texture 
    (CLUTTER_CLONE_TEXTURE(priv->real_texture))) == FALSE)
    {
      target_type = CGL_TEXTURE_RECTANGLE_ARB;
      cogl_enable (CGL_ENABLE_TEXTURE_RECT|CGL_ENABLE_BLEND);
    }
  else
    {
      target_type = CGL_TEXTURE_2D;
      cogl_enable (CGL_ENABLE_TEXTURE_2D|CGL_ENABLE_BLEND);
    }

  col.alpha = clutter_actor_get_opacity (self);
  cogl_color (&col);

  clutter_actor_get_coords (self, &x_1, &y_1, &x_2, &y_2);

  clone_texture_render_to_gl_quad (TIDY_PROXY_TEXTURE(self), 
				   0, 0, x_2 - x_1, y_2 - y_1);

  cogl_pop_matrix ();
}

static void
tidy_proxy_texture_request_coords (ClutterActor    *self,
				   ClutterActorBox *box)
{
  TidyProxyTexture         *proxy = TIDY_PROXY_TEXTURE(self);
  TidyProxyTexturePrivate  *priv = proxy->priv;  

  if (priv->real_texture)
    clutter_actor_request_coords (CLUTTER_ACTOR(priv->real_texture), box);

  CLUTTER_ACTOR_CLASS (tidy_proxy_texture_parent_class)->request_coords (self,
                                                                         box);
}

static void
tidy_proxy_texture_query_coords (ClutterActor        *self,
				 ClutterActorBox     *box)
{
  TidyProxyTexture         *proxy = TIDY_PROXY_TEXTURE(self);
  TidyProxyTexturePrivate  *priv = proxy->priv;  

  if (priv->real_texture)
    clutter_actor_query_coords (CLUTTER_ACTOR(priv->real_texture), box);
}

static void 
tidy_proxy_texture_dispose (GObject *object)
{
  TidyProxyTexture         *self = TIDY_PROXY_TEXTURE(object);
  TidyProxyTexturePrivate  *priv = self->priv;  

  if (priv->real_texture)
    g_object_unref (priv->real_texture);

  priv->real_texture = NULL;

  G_OBJECT_CLASS (tidy_proxy_texture_parent_class)->dispose (object);
}

static void 
tidy_proxy_texture_finalize (GObject *object)
{
  G_OBJECT_CLASS (tidy_proxy_texture_parent_class)->finalize (object);
}

static void
tidy_proxy_texture_set_property (GObject      *object,
				 guint         prop_id,
				 const GValue *value,
				 GParamSpec   *pspec)
{
  TidyProxyTexture *proxy = TIDY_PROXY_TEXTURE (object);

  switch (prop_id)
    {
    case PROP_PATH:
      tidy_proxy_texture_set_path (proxy, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
tidy_proxy_texture_get_property (GObject    *object,
				 guint       prop_id,
				 GValue     *value,
				 GParamSpec *pspec)
{
  TidyProxyTexture *proxy = TIDY_PROXY_TEXTURE (object);

  switch (prop_id)
    {
    case PROP_PATH:
      g_value_set_string (value, proxy->priv->path);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
tidy_proxy_texture_class_init (TidyProxyTextureClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  actor_class->paint = tidy_proxy_texture_paint;
  actor_class->request_coords = tidy_proxy_texture_request_coords;
  actor_class->query_coords   = tidy_proxy_texture_query_coords;

  gobject_class->finalize     = tidy_proxy_texture_finalize;
  gobject_class->dispose      = tidy_proxy_texture_dispose;
  gobject_class->set_property = tidy_proxy_texture_set_property;
  gobject_class->get_property = tidy_proxy_texture_get_property;

  g_object_class_install_property 
             (gobject_class,
	      PROP_PATH,
	      g_param_spec_string ("path",
				   "Path",
				   "Path to texture image",
				   NULL,
				   G_PARAM_READWRITE));

  g_type_class_add_private (gobject_class, sizeof (TidyProxyTexturePrivate));
}

static void
tidy_proxy_texture_init (TidyProxyTexture *self)
{
  TidyProxyTexturePrivate *priv;

  self->priv = priv = TIDY_PROXY_TEXTURE_GET_PRIVATE (self);

  priv->real_texture = NULL;
  priv->path = NULL;
}

static gboolean
tidy_proxy_texture_parse_custom_node (ClutterScriptable *scriptable,
                                      ClutterScript     *script,
                                      GValue            *value,
                                      const gchar       *name,
                                      JsonNode          *node)
{
  if (strcmp (name, "path") == 0)
    {
      const gchar *path;

      if (JSON_NODE_TYPE (node) != JSON_NODE_VALUE)
        return FALSE;

      path = json_node_get_string (node);
      if (!path || *path == '\0')
        return FALSE;

      g_value_init (value, G_TYPE_STRING);

      if (g_path_is_absolute (path))
        g_value_set_string (value, path);
      else
        {
          gboolean filename_set = FALSE;
          const gchar *filename = NULL;
          gchar *dir, *real_path;

          /* let's check if we have a filename */
          g_object_get (G_OBJECT (script),
                        "filename-set", &filename_set,
                        "filename", &filename,
                        NULL);

          if (filename_set)
            dir = g_path_get_dirname (filename);
          else
            dir = g_get_current_dir ();

          real_path = g_build_filename (dir, path, NULL);

          g_value_set_string (value, real_path);
          
          g_free (real_path);
          g_free (dir);
        }

      return TRUE;
    }

  return FALSE;
}

static void
clutter_scriptable_iface_init (ClutterScriptableIface *iface)
{
  iface->parse_custom_node = tidy_proxy_texture_parse_custom_node;
}

/**
 * tidy_proxy_texture_new:
 * @path: the path of the image to use
 *
 * Creates a proxy texture.
 *
 * Return value: the newly created #TidyProxyTexture
 */
ClutterActor *
tidy_proxy_texture_new (const gchar *path)
{
  g_return_val_if_fail (path != NULL, NULL);

  return g_object_new (TIDY_TYPE_PROXY_TEXTURE, "path", path, NULL);
}

G_CONST_RETURN gchar *
tidy_proxy_texture_get_path (TidyProxyTexture *proxy)
{
  g_return_val_if_fail (TIDY_IS_PROXY_TEXTURE (proxy), NULL);

  return proxy->priv->path;
}
