/* tidy-texture-reflection.c: adds a reflection on a texture
 *
 * Copyright (C) 2007 OpenedHand
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Author: Matthew Allum <mallum@o-hand.com>
 */

/**
 * SECTION:tidy-texture-reflection
 * @short_description: Creates a reflection of a texture
 *
 * #ClutterReflectTexture allows the cloning of existing #ClutterTexture with 
 * a reflection-like effect.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <clutter/cogl.h>
#include <clutter/clutter-util.h>

#include "tidy-texture-reflection.h"
#include "tidy-private.h"

enum
{
  PROP_0,

  PROP_REFLECTION_HEIGHT
};

#define TIDY_TEXTURE_REFLECTION_GET_PRIVATE(obj)        \
        (G_TYPE_INSTANCE_GET_PRIVATE ((obj),            \
         TIDY_TYPE_TEXTURE_REFLECTION,                  \
         TidyTextureReflectionPrivate))

struct _TidyTextureReflectionPrivate
{
  gint reflection_height;
};

G_DEFINE_TYPE (TidyTextureReflection,
               tidy_texture_reflection,
               CLUTTER_TYPE_CLONE_TEXTURE);

static void
reflect_texture_render_to_gl_quad (TidyTextureReflection *texture,
                                   ClutterTexture        *parent,
                                   gint                   x1, 
                                   gint                   y1,
                                   gint                   x2,
                                   gint                   y2)
{
  TidyTextureReflectionPrivate *priv = texture->priv;
  gint   qx1 = 0, qx2 = 0, qy1 = 0, qy2 = 0;
  gint   qwidth = 0, qheight = 0;
  gint   x, y, i =0, lastx = 0, lasty = 0;
  gint   n_x_tiles, n_y_tiles; 
  gint   pwidth, pheight, rheight;
  float tx, ty, ty2 = 0.0;
  ClutterColor white = { 255, 255, 255, 255 };

  qwidth  = x2 - x1;
  qheight = y2 - y1;

  rheight = priv->reflection_height;
  if (rheight < 0)
    rheight = clutter_actor_get_height (CLUTTER_ACTOR (parent)) / 2;

  if (rheight > qheight)
    rheight = qheight;

  /* Only paint if parent is in a state to do so */
  if (!clutter_texture_has_generated_tiles (parent))
    return;
  
  clutter_texture_get_base_size (parent, &pwidth, &pheight); 

  if (!clutter_texture_is_tiled (parent))
    {
      clutter_texture_bind_tile (parent, 0);

      /* NPOTS textures *always* used if extension available
       */
      if (clutter_feature_available (CLUTTER_FEATURE_TEXTURE_RECTANGLE))
	{
	  tx = (float) pwidth;
	  ty = (float) pheight;
	  ty2 = (float) (clutter_actor_get_height (CLUTTER_ACTOR (texture)) * rheight)
                        / pheight;
	  ty2 = pheight - ty2;

	}
      else
	{
	  tx = (float) pwidth / clutter_util_next_p2 (pwidth);  
	  ty = (float) pheight / clutter_util_next_p2 (pheight);
	}

      qx1 = x1; qx2 = x2;
      qy1 = y1; qy2 = y1 + rheight;

#ifdef CLUTTER_COGL_HAS_GL
      glBegin (GL_QUADS);

      white.alpha = clutter_actor_get_opacity (CLUTTER_ACTOR (texture));
      cogl_color (&white);

      glTexCoord2f (0, ty);   
      glVertex2i   (qx1, qy1);

      glTexCoord2f (tx,  ty);   
      glVertex2i   (qx2, qy1);

      white.alpha = 0;
      cogl_color (&white);

      glTexCoord2f (tx,  ty2);    
      glVertex2i   (qx2, qy2);
      
      glTexCoord2f (0, ty2);    
      glVertex2i   (qx1, qy2);

      glEnd ();	
#else
#warning "TidyTextureReflection does not currently support GL ES"
#endif
      return;
    }
  else
    {
      g_warning("TidyTextureReflection doesn't support tiled textures.."); 
    }

  /* FIXME: Below wont actually render the corrent graduated effect. */

#ifdef CLUTTER_COGL_HAS_GL
  clutter_texture_get_n_tiles (parent, &n_x_tiles, &n_y_tiles); 

  for (x = 0; x < n_x_tiles; x++)
    {
      lasty = 0;

      for (y = 0; y < n_y_tiles; y++)
	{
	  gint actual_w, actual_h;
	  gint xpos, ypos, xsize, ysize, ywaste, xwaste;
	  
	  clutter_texture_bind_tile (parent, i);
	 
	  clutter_texture_get_x_tile_detail (parent, x,
                                             &xpos, &xsize, &xwaste);
	  clutter_texture_get_y_tile_detail (parent, y,
                                             &ypos, &ysize, &ywaste);

	  actual_w = xsize - xwaste;
	  actual_h = ysize - ywaste;

	  tx = (float) actual_w / xsize;
	  ty = (float) actual_h / ysize;

	  qx1 = x1 + lastx;
	  qx2 = qx1 + ((qwidth * actual_w ) / pwidth );
	  
	  qy1 = y1 + lasty;
	  qy2 = qy1 + ((qheight * actual_h) / pheight );

	  glBegin (GL_QUADS);
	  glTexCoord2f (tx, ty);   glVertex2i   (qx2, qy2);
	  glTexCoord2f (0,  ty);   glVertex2i   (qx1, qy2);
	  glTexCoord2f (0,  0);    glVertex2i   (qx1, qy1);
	  glTexCoord2f (tx, 0);    glVertex2i   (qx2, qy1);
	  glEnd ();	

	  lasty += qy2 - qy1;	  

	  i++;
	}
      lastx += qx2 - qx1;
    }
#endif
}

static void
tidy_texture_reflection_paint (ClutterActor *actor)
{
#ifdef CLUTTER_COGL_HAS_GL
  TidyTextureReflection *texture;
  ClutterCloneTexture *clone;
  ClutterTexture *parent;
  ClutterActorBox box = { 0, };
  COGLenum target_type;
  ClutterColor white = { 255, 255, 255, 255 };

  texture = TIDY_TEXTURE_REFLECTION (actor);
  clone = CLUTTER_CLONE_TEXTURE (actor);

  parent = clutter_clone_texture_get_parent_texture (clone);
  if (!parent)
    return;

  if (!CLUTTER_ACTOR_IS_REALIZED (parent))
    clutter_actor_realize (CLUTTER_ACTOR (parent));

  if (clutter_feature_available (CLUTTER_FEATURE_TEXTURE_RECTANGLE) &&
      clutter_texture_is_tiled (parent) == FALSE)
    {
      target_type = CGL_TEXTURE_RECTANGLE_ARB;
      cogl_enable (CGL_ENABLE_TEXTURE_RECT | CGL_ENABLE_BLEND);
    }
  else
    {
      target_type = CGL_TEXTURE_2D;
      cogl_enable (CGL_ENABLE_TEXTURE_2D | CGL_ENABLE_BLEND);
    }

  cogl_push_matrix ();

  white.alpha = clutter_actor_get_opacity (actor);
  cogl_color (&white);

  clutter_actor_query_coords (actor, &box);

  reflect_texture_render_to_gl_quad (texture, parent,
                                     0, 0,
                                     CLUTTER_UNITS_TO_DEVICE (box.x2 - box.x1),
                                     CLUTTER_UNITS_TO_DEVICE (box.y2 - box.y1));

  cogl_pop_matrix ();
#endif
}

static void
tidy_texture_reflection_set_property (GObject      *gobject,
                                      guint         prop_id,
                                      const GValue *value,
                                      GParamSpec   *pspec)
{
  TidyTextureReflection *texture;

  texture = TIDY_TEXTURE_REFLECTION (gobject);

  switch (prop_id)
    {
    case PROP_REFLECTION_HEIGHT:
      tidy_texture_reflection_set_reflection_height (texture,
                                                     g_value_get_int (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
      break;
    }
}

static void
tidy_texture_reflection_get_property (GObject    *gobject,
                                      guint       prop_id,
                                      GValue     *value,
                                      GParamSpec *pspec)
{
  TidyTextureReflection *texture;

  texture = TIDY_TEXTURE_REFLECTION (gobject);

  switch (prop_id)
    {
    case PROP_REFLECTION_HEIGHT:
      g_value_set_int (value,
                       tidy_texture_reflection_get_reflection_height (texture));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop_id, pspec);
      break;
    }
}

static void
tidy_texture_reflection_class_init (TidyTextureReflectionClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (TidyTextureReflectionPrivate));

  gobject_class->set_property = tidy_texture_reflection_set_property;
  gobject_class->get_property = tidy_texture_reflection_get_property;

  actor_class->paint = tidy_texture_reflection_paint;

  g_object_class_install_property (gobject_class,
                                   PROP_REFLECTION_HEIGHT,
                                   g_param_spec_int ("reflection-height",
                                                     "Reflection Height",
                                                     "Height of the reflection, or -1",
                                                     -1, G_MAXINT, -1,
                                                     TIDY_PARAM_READWRITE));
}

static void
tidy_texture_reflection_init (TidyTextureReflection *texture)
{
  TidyTextureReflectionPrivate *priv;

  texture->priv = priv = TIDY_TEXTURE_REFLECTION_GET_PRIVATE (texture);
  
  priv->reflection_height = -1;
}

ClutterActor *
tidy_texture_reflection_new (ClutterTexture *texture)
{
  g_return_val_if_fail (texture == NULL || CLUTTER_IS_TEXTURE (texture), NULL);

  return g_object_new (TIDY_TYPE_TEXTURE_REFLECTION,
                       "parent-texture", texture,
                       NULL);
}

void
tidy_texture_reflection_set_reflection_height (TidyTextureReflection *texture,
                                               gint                   height)
{
  TidyTextureReflectionPrivate *priv;

  g_return_if_fail (TIDY_IS_TEXTURE_REFLECTION (texture));

  priv = texture->priv;

  if (priv->reflection_height != height)
    {
      priv->reflection_height = height;

      if (CLUTTER_ACTOR_IS_VISIBLE (texture))
        clutter_actor_queue_redraw (CLUTTER_ACTOR (texture));

      g_object_notify (G_OBJECT (texture), "reflection-height");
    }
}

gint
tidy_texture_reflection_get_reflection_height (TidyTextureReflection *texture)
{
  g_return_val_if_fail (TIDY_IS_TEXTURE_REFLECTION (texture), -1);

  return texture->priv->reflection_height;
}
