/* -*- Mode: C; c-file-style: "gnu"; tab-width: 8 -*- */
/* gtktimeline.c
 *
 * Copyright (C) 2007 Carlos Garnacho <carlos@imendio.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301 USA
 */

#include "gtytimeline.h"
#include "gtytypebuiltins.h"

#include <gtk/gtksettings.h>
#include <math.h>

#define GTY_TIMELINE_GET_PRIV(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), GTY_TYPE_TIMELINE, GtyTimelinePriv))
#define MSECS_PER_SEC 1000
#define FRAME_INTERVAL(nframes) (MSECS_PER_SEC / nframes)
#define DEFAULT_FPS 30

typedef struct GtyTimelinePriv GtyTimelinePriv;

struct GtyTimelinePriv
{
  guint duration;
  guint fps;
  guint source_id;

  GTimer *timer;

  gdouble progress;
  gdouble last_progress;

  GdkScreen *screen;

  guint animations_enabled : 1;
  guint loop               : 1;
  guint direction          : 1;
};

enum {
  PROP_0,
  PROP_FPS,
  PROP_DURATION,
  PROP_LOOP,
  PROP_DIRECTION,
  PROP_SCREEN
};

enum {
  STARTED,
  PAUSED,
  FINISHED,
  FRAME,
  LAST_SIGNAL
};

static guint signals [LAST_SIGNAL] = { 0, };


static void  gty_timeline_set_property  (GObject         *object,
                                         guint            prop_id,
                                         const GValue    *value,
                                         GParamSpec      *pspec);
static void  gty_timeline_get_property  (GObject         *object,
                                         guint            prop_id,
                                         GValue          *value,
                                         GParamSpec      *pspec);
static void  gty_timeline_finalize      (GObject *object);


G_DEFINE_TYPE (GtyTimeline, gty_timeline, G_TYPE_OBJECT)


static void
gty_timeline_class_init (GtyTimelineClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->set_property = gty_timeline_set_property;
  object_class->get_property = gty_timeline_get_property;
  object_class->finalize = gty_timeline_finalize;

  g_object_class_install_property (object_class,
				   PROP_FPS,
				   g_param_spec_uint ("fps",
						      "FPS",
						      "Frames per second for the timeline",
						      1, G_MAXUINT,
						      DEFAULT_FPS,
						      G_PARAM_READWRITE));
  g_object_class_install_property (object_class,
				   PROP_DURATION,
				   g_param_spec_uint ("duration",
						      "Animation Duration",
						      "Animation Duration",
						      0, G_MAXUINT,
						      0,
						      G_PARAM_READWRITE));
  g_object_class_install_property (object_class,
				   PROP_LOOP,
				   g_param_spec_boolean ("loop",
							 "Loop",
							 "Whether the timeline loops or not",
							 FALSE,
							 G_PARAM_READWRITE));
  g_object_class_install_property (object_class,
				   PROP_DIRECTION,
				   g_param_spec_enum ("direction",
						      "Direction",
						      "Whether the timeline moves forward or backward in time",
						      GTY_TYPE_TIMELINE_DIRECTION,
						      GTY_TIMELINE_DIRECTION_FORWARD,
						      G_PARAM_READWRITE));
  g_object_class_install_property (object_class,
				   PROP_SCREEN,
				   g_param_spec_object ("screen",
							"Screen",
							"Screen to get the settings from",
							GDK_TYPE_SCREEN,
							G_PARAM_READWRITE));

  signals[STARTED] =
    g_signal_new ("started",
		  G_TYPE_FROM_CLASS (object_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GtyTimelineClass, started),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  signals[PAUSED] =
    g_signal_new ("paused",
		  G_TYPE_FROM_CLASS (object_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GtyTimelineClass, paused),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  signals[FINISHED] =
    g_signal_new ("finished",
		  G_TYPE_FROM_CLASS (object_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GtyTimelineClass, finished),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

  signals[FRAME] =
    g_signal_new ("frame",
		  G_TYPE_FROM_CLASS (object_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GtyTimelineClass, frame),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__DOUBLE,
		  G_TYPE_NONE, 1,
		  G_TYPE_DOUBLE);

  g_type_class_add_private (klass, sizeof (GtyTimelinePriv));
}

static void
gty_timeline_init (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  priv->fps = DEFAULT_FPS;
  priv->duration = 0.0;
  priv->direction = GTY_TIMELINE_DIRECTION_FORWARD;
  priv->screen = gdk_screen_get_default ();

  priv->last_progress = 0;
}

static void
gty_timeline_set_property (GObject      *object,
                           guint         prop_id,
                           const GValue *value,
                           GParamSpec   *pspec)
{
  GtyTimeline *timeline;
  GtyTimelinePriv *priv;

  timeline = GTY_TIMELINE (object);
  priv = GTY_TIMELINE_GET_PRIV (timeline);

  switch (prop_id)
    {
    case PROP_FPS:
      gty_timeline_set_fps (timeline, g_value_get_uint (value));
      break;
    case PROP_DURATION:
      gty_timeline_set_duration (timeline, g_value_get_uint (value));
      break;
    case PROP_LOOP:
      gty_timeline_set_loop (timeline, g_value_get_boolean (value));
      break;
    case PROP_DIRECTION:
      gty_timeline_set_direction (timeline, g_value_get_enum (value));
      break;
    case PROP_SCREEN:
      gty_timeline_set_screen (timeline,
                               GDK_SCREEN (g_value_get_object (value)));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
gty_timeline_get_property (GObject    *object,
                           guint       prop_id,
                           GValue     *value,
                           GParamSpec *pspec)
{
  GtyTimeline *timeline;
  GtyTimelinePriv *priv;

  timeline = GTY_TIMELINE (object);
  priv = GTY_TIMELINE_GET_PRIV (timeline);

  switch (prop_id)
    {
    case PROP_FPS:
      g_value_set_uint (value, priv->fps);
      break;
    case PROP_DURATION:
      g_value_set_uint (value, priv->duration);
      break;
    case PROP_LOOP:
      g_value_set_boolean (value, priv->loop);
      break;
    case PROP_DIRECTION:
      g_value_set_enum (value, priv->direction);
      break;
    case PROP_SCREEN:
      g_value_set_object (value, priv->screen);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
gty_timeline_finalize (GObject *object)
{
  GtyTimelinePriv *priv;

  priv = GTY_TIMELINE_GET_PRIV (object);

  if (priv->source_id)
    {
      g_source_remove (priv->source_id);
      priv->source_id = 0;
    }

  if (priv->timer)
    g_timer_destroy (priv->timer);

  G_OBJECT_CLASS (gty_timeline_parent_class)->finalize (object);
}

static gboolean
gty_timeline_run_frame (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;
  gdouble delta_progress, progress;
  guint elapsed_time;

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  elapsed_time = (guint) (g_timer_elapsed (priv->timer, NULL) * 1000);
  g_timer_start (priv->timer);

  if (priv->animations_enabled)
    {
      delta_progress = (gdouble) elapsed_time / priv->duration;
      progress = priv->last_progress;

      if (priv->direction == GTY_TIMELINE_DIRECTION_BACKWARD)
	progress -= delta_progress;
      else
	progress += delta_progress;

      priv->last_progress = progress;

      progress = CLAMP (progress, 0., 1.);
    }
  else
    progress = (priv->direction == GTY_TIMELINE_DIRECTION_FORWARD) ? 1.0 : 0.0;

  priv->progress = progress;
  g_signal_emit (timeline, signals [FRAME], 0, progress);

  if ((priv->direction == GTY_TIMELINE_DIRECTION_FORWARD && progress == 1.0) ||
      (priv->direction == GTY_TIMELINE_DIRECTION_BACKWARD && progress == 0.0))
    {
      if (!priv->loop)
	{
	  if (priv->source_id)
	    {
	      g_source_remove (priv->source_id);
	      priv->source_id = 0;
	    }
          g_timer_stop (priv->timer);
	  g_signal_emit (timeline, signals [FINISHED], 0);
	  return FALSE;
	}
      else
	gty_timeline_rewind (timeline);
    }

  return TRUE;
}

/**
 * gty_timeline_new:
 * @duration: duration in milliseconds for the timeline
 *
 * Creates a new #GtyTimeline with the specified number of frames.
 *
 * Return Value: the newly created #GtyTimeline
 **/
GtyTimeline *
gty_timeline_new (guint duration)
{
  return g_object_new (GTY_TYPE_TIMELINE,
		       "duration", duration,
		       NULL);
}

GtyTimeline *
gty_timeline_new_for_screen (guint      duration,
                             GdkScreen *screen)
{
  return g_object_new (GTY_TYPE_TIMELINE,
		       "duration", duration,
		       "screen", screen,
		       NULL);
}

/**
 * gty_timeline_start:
 * @timeline: A #GtyTimeline
 *
 * Runs the timeline from the current frame.
 **/
void
gty_timeline_start (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;
  gboolean enable_animations = FALSE;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (!priv->source_id)
    {
      if (priv->timer)
        g_timer_continue (priv->timer);
      else
        priv->timer = g_timer_new ();

      /* sanity check */
      g_assert (priv->fps > 0);

      if (priv->screen)
        {
#if 0
          GtkSettings *settings = gtk_settings_get_for_screen (priv->screen);
          g_object_get (settings, "gty-enable-animations", &enable_animations, NULL);
#else
          // XXX
          enable_animations = TRUE;
#endif
        }

      priv->animations_enabled = (enable_animations == TRUE);

      g_signal_emit (timeline, signals [STARTED], 0);

      if (enable_animations)
        priv->source_id = gdk_threads_add_timeout (FRAME_INTERVAL (priv->fps),
                                                   (GSourceFunc) gty_timeline_run_frame,
                                                   timeline);
      else
        priv->source_id = gdk_threads_add_idle ((GSourceFunc) gty_timeline_run_frame,
                                                timeline);
    }
}

/**
 * gty_timeline_pause:
 * @timeline: A #GtyTimeline
 *
 * Pauses the timeline.
 **/
void
gty_timeline_pause (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (priv->source_id)
    {
      g_timer_stop (priv->timer);
      g_source_remove (priv->source_id);
      priv->source_id = 0;
      g_signal_emit (timeline, signals [PAUSED], 0);
    }
}

/**
 * gty_timeline_rewind:
 * @timeline: A #GtyTimeline
 *
 * Rewinds the timeline.
 **/
void
gty_timeline_rewind (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (gty_timeline_get_direction(timeline) != GTY_TIMELINE_DIRECTION_FORWARD)
    priv->progress = priv->last_progress = 1.;
  else
    priv->progress = priv->last_progress = 0.;

  /* reset timer */
  if (priv->timer)
    {
      g_timer_start (priv->timer);

      if (!priv->source_id)
        g_timer_stop (priv->timer);
    }
}

/**
 * gty_timeline_is_running:
 * @timeline: A #GtyTimeline
 *
 * Returns whether the timeline is running or not.
 *
 * Return Value: %TRUE if the timeline is running
 **/
gboolean
gty_timeline_is_running (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), FALSE);

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  return (priv->source_id != 0);
}

/**
 * gty_timeline_get_fps:
 * @timeline: A #GtyTimeline
 *
 * Returns the number of frames per second.
 *
 * Return Value: frames per second
 **/
guint
gty_timeline_get_fps (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), 1);

  priv = GTY_TIMELINE_GET_PRIV (timeline);
  return priv->fps;
}

/**
 * gty_timeline_set_fps:
 * @timeline: A #GtyTimeline
 * @fps: frames per second
 *
 * Sets the number of frames per second that
 * the timeline will play.
 **/
void
gty_timeline_set_fps (GtyTimeline *timeline,
                      guint        fps)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));
  g_return_if_fail (fps > 0);

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  priv->fps = fps;

  if (gty_timeline_is_running (timeline))
    {
      g_source_remove (priv->source_id);
      priv->source_id = gdk_threads_add_timeout (FRAME_INTERVAL (priv->fps),
						 (GSourceFunc) gty_timeline_run_frame,
						 timeline);
    }

  g_object_notify (G_OBJECT (timeline), "fps");
}

/**
 * gty_timeline_get_loop:
 * @timeline: A #GtyTimeline
 *
 * Returns whether the timeline loops to the
 * beginning when it has reached the end.
 *
 * Return Value: %TRUE if the timeline loops
 **/
gboolean
gty_timeline_get_loop (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), FALSE);

  priv = GTY_TIMELINE_GET_PRIV (timeline);
  return priv->loop;
}

/**
 * gty_timeline_set_loop:
 * @timeline: A #GtyTimeline
 * @loop: %TRUE to make the timeline loop
 *
 * Sets whether the timeline loops to the beginning
 * when it has reached the end.
 **/
void
gty_timeline_set_loop (GtyTimeline *timeline,
                       gboolean     loop)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (loop != priv->loop)
    {
      priv->loop = loop;
      g_object_notify (G_OBJECT (timeline), "loop");
    }
}

void
gty_timeline_set_duration (GtyTimeline *timeline,
                           guint        duration)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (duration != priv->duration)
    {
      priv->duration = duration;
      g_object_notify (G_OBJECT (timeline), "duration");
    }
}

guint
gty_timeline_get_duration (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), 0);

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  return priv->duration;
}

/**
 * gty_timeline_set_direction:
 * @timeline: A #GtyTimeline
 * @direction: direction
 *
 * Sets the direction of the timeline.
 **/
void
gty_timeline_set_direction (GtyTimeline          *timeline,
                            GtyTimelineDirection  direction)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (direction != priv->direction)
    {
      priv->direction = direction;
      g_object_notify (G_OBJECT (timeline), "direction");
    }
}

/**
 * gty_timeline_get_direction:
 * @timeline: A #GtyTimeline
 *
 * Returns the direction of the timeline.
 *
 * Return Value: direction
 **/
GtyTimelineDirection
gty_timeline_get_direction (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), GTY_TIMELINE_DIRECTION_FORWARD);

  priv = GTY_TIMELINE_GET_PRIV (timeline);
  return priv->direction;
}

void
gty_timeline_set_screen (GtyTimeline *timeline,
                         GdkScreen   *screen)
{
  GtyTimelinePriv *priv;

  g_return_if_fail (GTY_IS_TIMELINE (timeline));
  g_return_if_fail (GDK_IS_SCREEN (screen));

  priv = GTY_TIMELINE_GET_PRIV (timeline);

  if (priv->screen)
    g_object_unref (priv->screen);

  priv->screen = g_object_ref (screen);

  g_object_notify (G_OBJECT (timeline), "screen");
}

GdkScreen *
gty_timeline_get_screen (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), NULL);

  priv = GTY_TIMELINE_GET_PRIV (timeline);
  return priv->screen;
}

gdouble
gty_timeline_get_progress (GtyTimeline *timeline)
{
  GtyTimelinePriv *priv;

  g_return_val_if_fail (GTY_IS_TIMELINE (timeline), 0.);

  priv = GTY_TIMELINE_GET_PRIV (timeline);
  return priv->progress;
}

gdouble
gty_timeline_calculate_progress (gdouble                 linear_progress,
                                 GtyTimelineProgressType progress_type)
{
  gdouble progress;

  progress = linear_progress;

  switch (progress_type)
    {
    case GTY_TIMELINE_PROGRESS_LINEAR:
      break;
    case GTY_TIMELINE_PROGRESS_SINUSOIDAL:
      progress = sinf ((progress * G_PI) / 2);
      break;
    case GTY_TIMELINE_PROGRESS_EXPONENTIAL:
      progress *= progress;
      break;
    case GTY_TIMELINE_PROGRESS_EASE_IN_EASE_OUT:
      {
        progress *= 2;

        if (progress < 1)
          progress = pow (progress, 3) / 2;
        else
          progress = (pow (progress - 2, 3) + 2) / 2;
      }
    }

  return progress;
}
